/*
 * uart_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * The vUARTTask first configures the UART 2 port to run from the 16 MHZ PIOSC
 * and output data with a configuration of 115,200, 8-N-1.  Next, a mutex is
 * created and only if successful, the two data output tasks are created after.
 *
 * The prvUARTWrite function takes the mutex and uses a basic loop that scans
 * for the end of string null character while sending out data one bye at a
 * time with UARTCharPut.  The function includes a variable delay to simulate
 * an application where the UART output is event driven and not an endless
 * stream.  After the UART message has been sent, it gives back the mutex.
 *
 * The prvUARTPrintf function takes the mutex and processes any variable
 * arguments that are input to support the advanced functionality of the
 * UARTvprintf to display specific character types.  Then UARTvprintf is called
 * with the string parameter and the variable arguments being passed to it.
 * The function includes a variable delay to simulate an application where the
 * UART output is event driven and not an endless stream.  After the UART
 * message has been sent, it gives back the mutex.  For a full list of valid
 * variable arguments, refer to the UARTvprintf function in uartstdio.c.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "drivers/rtos_hw_drivers.h"
#include "utils/uartstdio.h"
#include "uart_task.h"
/*-----------------------------------------------------------*/

/*
 * The mutex used to guard the UART resource between tasks.
 */
SemaphoreHandle_t xMutex = NULL;

/*
 * Called by main() to setup the UART for output.
 */
void vUARTConfigure( void );

/*
 * Configure UART2 to send messages to a terminal.
 */
static void prvConfigureUART( void );
/*-----------------------------------------------------------*/

void vUARTConfigure( void )
{
    /* Configure the UART 2 peripheral. */
    prvConfigureUART();

    /* Before a semaphore is used it must be explicitly created. In this example a
     * mutex type semaphore is created. */
    xMutex = xSemaphoreCreateMutex();
}
/*-----------------------------------------------------------*/

void prvUARTWrite( const char *pcString )
{
uint32_t ui32Idx;

    /* The mutex is created before the scheduler is started, so already exists
     * by the time this task executes.  Attempt to take the mutex, blocking
     * indefinitely to wait for the mutex if it is not available right away. */
    if ( xSemaphoreTake( xMutex, portMAX_DELAY ) == pdTRUE )
    {
        /* Loop through until reaching the end of the string. */
        for(ui32Idx = 0; pcString[ui32Idx] != '\0'; ui32Idx++)
        {
            /* Output the full string. */
            UARTCharPut(UART2_BASE, pcString[ui32Idx]);
        }
    }

    /* The mutex MUST be given back. */
    xSemaphoreGive( xMutex );
}
/*-----------------------------------------------------------*/

void prvUARTPrintf( const char *pcString, ... )
{
/* vaArgP is a variable argument list pointer whose content will depend
 * upon the format string passed in pcString. */
va_list vaArgP;

    /* The mutex is created before the scheduler is started, so already exists
     * by the time this task executes.  Attempt to take the mutex, blocking
     * indefinitely to wait for the mutex if it is not available right away. */
    if ( xSemaphoreTake( xMutex, portMAX_DELAY ) == pdTRUE )
    {
        /* Process any passed variable arguments. */
        va_start(vaArgP, pcString);

        /* Call the UART standard IO API to output the UART message by
         * providing both the string and any received variable arguments. */
        UARTvprintf(pcString, vaArgP);

        /* After the UART message is output, the variable arguments are no
         * longer needed. */
        va_end(vaArgP);
    }

    /* The mutex MUST be given back. */
    xSemaphoreGive( xMutex );
}
/*-----------------------------------------------------------*/

static void prvConfigureUART( void )
{
    /* Enable GPIO port D which is used for UART2 pins.
     * TODO: change this to whichever GPIO port you are using. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);

    /* Configure the pin muxing for UART2 functions on port D4 and D5.
     * TODO: change this to select the port/pin you are using. */
    GPIOPinConfigure(GPIO_PD4_U2RX);
    GPIOPinConfigure(GPIO_PD5_U2TX);

    /* Enable UART2 so that we can configure the clock. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART2);

    /* Use the internal 16MHz oscillator as the UART clock source. */
    UARTClockSourceSet(UART2_BASE, UART_CLOCK_PIOSC);

    /* Select the alternate (UART) function for these pins.
     * TODO: change this to select the port/pin you are using. */
    GPIOPinTypeUART(GPIO_PORTD_BASE, GPIO_PIN_4 | GPIO_PIN_5);

    /* Initialize the UART2 for console I/O. */
    UARTStdioConfig(2, 115200, 16000000);
}
/*-----------------------------------------------------------*/
